# Appointments App

A comprehensive appointment scheduling system for Django applications. This app provides functionality for managing appointments, services, staff members, and business hours.

## Features

- **Appointment Management**: Schedule, view, update, and cancel appointments
- **Service Management**: Define and manage different services with durations and prices
- **Staff Management**: Assign staff members to specific services and manage their availability
- **Business Hours**: Set working hours for each day of the week
- **Notes**: Add notes to appointments for internal use
- **REST API**: Full-featured API for integration with other systems
- **Admin Interface**: Comprehensive admin interface for managing all aspects of the system

## Installation

1. Add 'appointments' to your INSTALLED_APPS setting:

```python
INSTALLED_APPS = [
    # ...
    'appointments',
    'rest_framework',
]
```

2. Run migrations:
```bash
python manage.py makemigrations
python manage.py migrate
```

3. Include the URL configuration in your project's urls.py:

```python
from django.urls import include, path

urlpatterns = [
    # ...
    path('appointments/', include('appointments.urls')),
]
```

## Configuration

### Required Settings

Add these settings to your Django settings file:

```python
# Settings for the appointments app
APPOINTMENT_CONFIG = {
    # Time slot interval in minutes (default: 15)
    'TIME_SLOT_INTERVAL': 15,
    
    # Minimum time in hours before an appointment can be canceled (default: 2)
    'MIN_CANCEL_HOURS': 2,
    
    # Default timezone for the business
    'DEFAULT_TIMEZONE': 'UTC',
    
    # Email settings for notifications
    'EMAIL_NOTIFICATIONS': {
        'enabled': True,
        'from_email': 'noreply@example.com',
        'staff_notification_emails': ['staff@example.com'],
    },
}
```

### Optional Settings

- `APPOINTMENT_CALENDAR_DAYS`: Number of days to show in the calendar view (default: 30)
- `APPOINTMENT_PAGINATION`: Number of items per page in list views (default: 20)

## Usage

### Templates

Include these template tags in your base template:

```html
{% load static %}

<!-- Include CSS -->
<link rel="stylesheet" href="{% static 'appointments/css/appointments.css' %}">

<!-- Include JavaScript -->
<script src="{% static 'appointments/js/appointments.js' %}"></script>
```

### API Endpoints

The following API endpoints are available:

- `GET /api/appointments/`: List all appointments
- `POST /api/appointments/`: Create a new appointment
- `GET /api/appointments/{id}/`: Retrieve a specific appointment
- `PUT /api/appointments/{id}/`: Update an appointment
- `DELETE /api/appointments/{id}/`: Delete an appointment
- `GET /api/available-slots/`: Get available time slots
- `GET /api/services/`: List all services
- `GET /api/staff/`: List all staff members

## Signals

The app provides the following signals:

- `appointment_created`: Sent when a new appointment is created
- `appointment_updated`: Sent when an appointment is updated
- `appointment_cancelled`: Sent when an appointment is cancelled
- `appointment_reminder`: Sent as a reminder before an appointment

## Testing

Run the test suite with:

```bash
python manage.py test appointments
```

## Contributing

1. Fork the repository
2. Create your feature branch (`git checkout -b feature/AmazingFeature`)
3. Commit your changes (`git commit -m 'Add some AmazingFeature'`)
4. Push to the branch (`git push origin feature/AmazingFeature`)
5. Open a Pull Request

## License

This project is licensed under the MIT License - see the LICENSE file for details.
